%% Implementation of CM decomposition with S matrix (included in AToM)
%
% Works with MATLAB R2017a+
% Needs installed and fully operational AToM, download: antennatoolbox.com
% 
% Miloslav Capek, miloslav.capek@fel.cvut.cz
% 
% Theory is derived and published in:
% Tayli, D., Capek, M., Akrou, L., Losenicky, V., Jelinek, L., Gustafsson, M.: 
% Accurate and Efficient Evaluation of Characteristic Modes

clear;
clc;
close all;

Nmodes = 500; % say how many modes should be calculated

%% Generate mesh grid for sphere using Lebedev sphere
Ns = 590;
Sc = [1 1 1];
leb = getLebedevSphere(Ns);
x = Sc(1)*leb.x;
y = Sc(2)*leb.y;
z = Sc(3)*leb.z;
DT = delaunayTriangulation(x(:),y(:),z(:));

Mesh = models.utilities.meshPublic.getMeshData2D(DT.Points, DT.convexHull);

ka = 1/2;
a = models.utilities.meshPublic.getCircumsphere(Mesh.nodes);
f = models.utilities.converter.k0tof0(ka/a);

%% Settings for AToM MoM
settings = struct(...
    'quadOrder', 3, ...
    'nBatchMax', 5e3, ...
    'nWorkers', 0, ...
    'resultPath', pwd, ...
    'resultsInRAM', true, ...
    'resultsInFile', false, ...
    'verbosity', 2);
requests = {'mesh', 'basisFcns', 'zMat'};

%% MoM calculation
fprintf(1, 'Calculation of Z-matrix operator... \n');
Res = models.solvers.MoM2D.solve(...
    Mesh, f, [], [], [1 1 1], settings, requests, []);

BF = Res.basisFcns;
R  = real(Res.zMat.data);
X  = imag(Res.zMat.data);

fprintf(1, 'Calculation of S-matrix operator... \n');
Lmax = ceil(ka + 7*ka^(1/3) + 3);
[S, indexMatrix] = models.utilities.matrixOperators.SMatrix.computeS(...
    Mesh, BF, f, Lmax + 5, 3);
nCMmax = min(size(S));

%% Normalization of matrix operators
% EdgesLength = ones(BF.nUnknowns, 1);
EdgesLength = Mesh.triangleEdgeLengths(BF.data(:,3));
lmm  = repmat(EdgesLength.', size(S, 1), 1);
lmln = (EdgesLength*EdgesLength');
Sn   = S./lmm;
Rn   = R./lmln;
Xn   = X./lmln;

%% Classical CM
t = tic;
[V, N] = eigs(X, R, Nmodes, 'sm');
N = diag(N);
tc = toc(t);
fprintf(1, 'o (1) Classical (X, R) EIGS done, %5.5fs\n', tc);

%% Novel CM
% Please, cite: 
% Tayli, D., Capek, M., Akrou, L., Losenicky, V., Jelinek, L., Gustafsson, M.: 
% Accurate and Efficient Evaluation of Characteristic Modes
t = tic;
[SvecU, Ssvd, SvecV] = svd(Sn);
Ssvd_square = Ssvd(1:nCMmax,1:nCMmax);
% SVD-representation of the problem
Xq  = SvecV.'*Xn*SvecV;
RU1 = Ssvd_square*Ssvd_square.';
% Model-order reduction:
X11 = Xq(1:nCMmax, 1:nCMmax);
X12 = Xq(1:nCMmax, nCMmax+1:end);
X21 = Xq(nCMmax+1:end, 1:nCMmax);
X22 = Xq(nCMmax+1:end, nCMmax+1:end);
X22iX21 = X22\X21;
XU1 = X11 - X12*X22iX21;
% Generalized problem from SVD
[Vq, Nq] = eigs(XU1, RU1, min(Nmodes, nCMmax-1), 'sm');
Nq = diag(Nq);
% Recalc. data back
Vq = SvecV*[eye(size(Vq,1)); -X22iX21]*Vq;
% Sort data
[~, iq] = sort(abs(real(Nq)));
Nq = Nq(iq);
Vq = Vq(:, iq);
tn1 = toc(t);
fprintf(1, 'o (2) Novel (V*X*V'',A*A'') EIGS done, %5.5fs\n', tn1);

%% Super-fast version to get eigen-numbers
t = tic;
invXS = Xn \ Sn';
[Ixs, Nxs] = eig(Sn*invXS);
Nxs = 1./diag(Nxs);
Iss = (invXS*Ixs)*diag(Nxs); % offdiagonal terms in Nss are inf
tn2 = toc(t);
fprintf(1, 'o (3) Novel (Sn*Xn\\Sn'') EIGS done, %5.5fs\n', tn2);
% currents here are not normalized to unitary radiated power

%% Exact results for sphere
nLambda  = Lmax + 1;
multipl  = 2*(1:nLambda)+1;
n        = 1:nLambda;
lambdaTE = - (bessely(n + 0.5, ka))./(besselj(n + 0.5, ka));
lambdaTM = -(ka.*bessely(n - 0.5, ka) - n.*bessely(n + 0.5, ka)) ./ ...
    (ka.*besselj(n - 0.5, ka) - n.*besselj(n + 0.5, ka));
Lambda   = [repelem(lambdaTE, multipl) repelem(lambdaTM, multipl)];

%% Plot results
figure('color', 'w', 'pos', [50 50 1250 750]);
Nex = abs(sort(real(Lambda)));
iex = find(min(Nex) == Nex, 1, 'last');
Nqs = abs(sort(real(Nq)));
iqs = find(min(Nqs) == Nqs, 1);
Ns  = abs(sort(real(N)));
is  = find(min(Ns) == Ns, 1);
Nss = abs(sort(real(Nxs)));
iss = find(min(Nss) == Nss, 1);
semilogy((1:length(Nex))-iex, Nex, 'k-', 'LineWidth', 1/2);
hold on; grid on;
semilogy((1:length(Ns))-is, Ns, 'r--');
semilogy((1:length(Nqs))-iqs, Nqs, 'x', 'MarkerSize', 10);
semilogy((1:length(Nss))-iss, Nss, '*g', 'MarkerSize', 5);
xlabel('modes');
ylabel('|\lambda_n|');
title('Comparison of char. numbers calculate by various methods');
legend('analytics', ...
    ['Classical eigs(X,R), ' num2str(round(tc, 2)) 's'], ...
    ['Novel method 1, ' num2str(round(tn1, 2)) 's'], ...
    ['Novel method 2, ' num2str(round(tn2, 2)) 's']);
set(gca, 'FontSize', 14);

%% Show current
results.plotCurrent(Mesh, 'basisFcns', BF, 'iVec', Iss(:,1), ...
    'arrowScale', 'proportional');