function Mesh = pixelGridToMesh(M, d)
%% pixelGridToMesh Delivers Mesh from matrix full of integer numbers
%  Each entry is considered as a pixel of size d (by default d = 1)
% 
%  INPUTS
%     M: matrix of integer numbers denoting type of mesh in that pixel
% 
%  INPUTS
%   (optional)
%     d: size of the pixel (by default d = 1)
% 
%  OUTPUTS
%     Mesh: mesh grid in AToM format
% 
%  SYNTAX
%
%  Mesh = meshFromPixels(ones(10, 5), 1);
%
%  Type of elementary mesh cells:
%   M(i,j) = +1 pixel has two triangles with boundary diagonal /
%   M(i,j) = -1 pixel has two triangles with boundary diagonal \
%   M(i,j) = +2 pixel has four triangles devided by two diagonals X
% 
% Included in AToM, info@antennatoolbox.com
%  2018, Miloslav Capek, CTU in Prague, miloslav.capek@fel.cvut.cz
% mcode

if nargin < 2
    d = 1;
end

%% Prepare elementary cells
A = [  0,   0, 0];
B = [  d,   0, 0];
C = [  d,   d, 0];
D = [  0,   d, 0];
E = [d/2, d/2, 0];

% P has +1 in M matrix
pP = [A; B; C; D];
tP = [1 2 3; 1 3 4];
% L has -1 in M matrix
pL = [A; B; C; D];
tL = [1 2 4; 2 3 4];
% C has 2 in M matrix
pC = [A; B; C; D; E];
tC = [1 2 5; 2 3 5; 3 4 5; 4 1 5];
% (hole has 0 in M matrix)

%% Generate mesh
p = double.empty(0,3);
t = double.empty(0,3);
nM = size(M);
h = waitbar(0,'AToM prepares your pixelized structure...');
for m1 = 0:(nM(1)-1)
    for m2 = 0:(nM(2)-1)
        pn = double.empty(0,3);
        tn = double.empty(0,3);
        if M(m1+1,m2+1) == 1
            pn = pP + repmat([m2 m1 0], size(pP, 1), 1);
            tn = tP;
        elseif M(m1+1,m2+1) == -1
            pn = pL + repmat([m2 m1 0], size(pL, 1), 1);
            tn = tL;            
        elseif M(m1+1,m2+1) == 2
            pn = pC + repmat([m2 m1 0], size(pC, 1), 1);
            tn = tC;
        end
        [p, t] = models.utilities.meshPublic.uniteMeshes(p, t, pn, tn);
    end
    waitbar(m1 / (nM(1)-1));
end
close(h);

%% Get Mesh structure
Mesh = models.utilities.meshPublic.getMeshData2D(p, t);